#Livre : La régression logistique en épidémiologie
#----------------------------------------------------------------------------------------------------
# Programme de calcul des OR par classes après modélisation par polynômes fractionnaires (fonction principalement utilisé dans le chapitre 4 du livre)
#
# NB : il existe une autre fonction utilisable après une transformation par fonctions splines (ORcl_sp)
# ----------------------------------------------------------------------------------------------------
#
# Ce programme est appelé par la fonction ORcl_pf après une transformation de la variable X par des polynômes fractionnaires réalisée avec mfp() du package mfp.
#
# Les arguments de la fonction sont :  ORcl.pf(x,res.mfp,ref,cl)
#
# - x = nom de la variable transformée en polynômes fractionnaires (à mettre entre "")
# - res.mfp : résultat du modèle du modèle obtenu par mfp
# - ref : référence (en général, centre de la classe de référence)
# - cl : valeurs pour lesquelles on veut les OR par rapport à ref (en général, centres des classes pour lesquelles on veut les OR). Doit être mis sous la forme cl=c(x1,x2,...)
#
#
# Exemple avec le fichier cycles3 utilisé dans le chapitre 4
#
# fp2 <-mfp(acc~fp(age+ovo,df=4,scale=T),family=binomial(),data=cycles3,select=0.05,verbose=TRUE)
# ORcl_pf(x="age",res.mfp=fp2,ref=27,cl=c(17,22,32,37,42))
# 
# Résultats
# 
# Modélisation : mfp(formula = acc ~ fp(age, df = 4, scale = FALSE), data = cycles3,     family = binomial(), select = 0.05, verbose = TRUE) 
# Puissance(s) de la variable age transformée en polynômes fractionnaires : 3 3 
# 
# OR et IC pour la variable age 
# 
# ref =  27  classe =  17  OR =  0.57  95% IC : [ 0.38  -  0.86 ]
# ref =  27  classe =  22  OR =  0.80  95% IC : [ 0.66  -  0.98 ]
# ref =  27  classe =  32  OR =  0.96  95% IC : [ 0.85  -  1.10 ]
# ref =  27  classe =  37  OR =  0.61  95% IC : [ 0.51  -  0.72 ]
# ref =  27  classe =  42  OR =  0.21  95% IC : [ 0.15  -  0.29 ]
# ----------------------------------------------------------------------------------------------------

ORcl_pf <- function(x,res.mfp,ref,cl)  {

library(mfp)

# 1.Vérification des bonnes conditions de la commande

# 1.1 Contrôle du nom de la variable transformée en PF
tok <- strsplit(as.character(res.mfp[["call"]]),"\\(")
tok2 <- strsplit(tok[[2]],",")
if(tok2[[2]][1]!=x) {
  stop (paste("Ce n'est pas la variable",x,"qui est transformée en polynôme fractionnaire"))
}
# 1.2 contrôle de res.mfp
if (!exists("res.mfp")) {
  stop("L'objet res.mfp n'existe pas.")
}
# 1.3 Limitation à des PF de degré 2
if (fp2[["df.final"]][x,"df.final"]>4)  {
	stop("le nombre de puissance pour la variable doit être ≤ 2, c'est-à-dire df ≤ 4")
}
# 1.4 La liste des classes ne contient pas ref
if (ref %in% cl) {
  stop("ref ne doit pas faire partie de cl")
}


# 2. Récupération de valeurs
  
# 2.1 Catégorie de référence et répercution du changement d'échelle de mfp
ref_or <- ref # concervation de la valeur d'origine de ref pour l'affichage des résultats
ref <- (ref_or - fp2[["scale"]][x,"shift"])/fp2[["scale"]][x,"scale"]  # pour tenir compte de l'échelle sur X

# 2.2 Puissances de X
p1 <- res.mfp[["powers"]][as.character(x),"power1"] 
p2 <- res.mfp[["powers"]][as.character(x),"power2"]


# 3. Annonce des résultats (mise ici pour ne pas être répétée à cause des boucles ultérieures)
modélisation <- paste(deparse(fp2[["call"]]), collapse = "")
cat("Modélisation :",modélisation,"\n")
cat("Puissance(s) de la variable",x,"transformée en polynômes fractionnaires :" ,p1,p2,"\n")
cat("\nOR et IC pour la variable",x,"\n\n")


# 4. Calcul des coefficients qui doivent figurer dans lincom. Il y en a 1 ou 2 par valeurs de cl (car 2 puissances au plus pour la variable X). Il faut donc une boucle sur les valeurs de cl
# nb : la 2ème puissance (`2') est toujours inférieure ou égale à la 1ère (`1'). D'où les if et else successifs au sein de la boucle.

for (i in cl) {  # Valeurs des polynômes fractionnaires pour la référence et pour la classe (i)
  # 4.1 mise à l'échelle des valeurs de cl et conservation des valeurs d'origine pour l'affichage des résultats
  i_or <- i  
  i <- (i-fp2[["scale"]][x,"shift"])/fp2[["scale"]][x,"scale"]
  
  # 4.2 Calcul des 2 coefficients d1 et d2
  if (p1!=0) {
  d1=(i)^p1-(ref)^p1
  } 
  else {
  d1=log(i)-log(ref)
	}
	if (is.na(p2)) {
	  d2=0
	}
	else if (!is.na(p2) & p2!= p1) {
		d2=(i)^p2-(ref)^p2
	}
	else if (p2!=0 & p2==p1) {
		d2=(i)^p1*log(i)-(ref)^p1*log(ref)
	}
	else if (p2==0 & p1==0) {
		d2=log(i)*log(i)-log(ref)*log(ref)
	}
	

#	5. Résultat (avec 2 décimales pour les OR et IC)
	
	if (!is.na(p2)) {
	formule.lincom <- paste0(d1,"*",x,".1 + ",d2,"*",x,".2")
	} 
  else {
	  formule.lincom <- paste0(d1,"*",x,".1")
	}
	 
# 5.1 Application de lincom avec la formule
combi.lin <- lincom(fp2, formule.lincom, eform = TRUE)

# 5.2 Affichage des résultats
cat("ref = ", ref_or, " classe = ", i_or, 
    " OR = ", sprintf("%.2f", combi.lin$Estimate),
    " 95% IC : [", sprintf("%.2f", combi.lin$`2.5 %`), 
    " - ", sprintf("%.2f", combi.lin$`97.5 %`), "]\n")

}  # fin de boucle i sur les valeurs de cl
  

}  # fin du programme